/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Diagnostics;
using System.Data;
using System.Runtime.Serialization;
using System.Globalization;
using System.ComponentModel;

using Borland.Eco.Persistence;
using Borland.Eco.Persistence.Default;
using Borland.Eco.Persistence.Configuration;

namespace Borland.Eco.Persistence.Default
{
	[Serializable]
	public class InvalidFieldTypeException: Exception 
	{
		public InvalidFieldTypeException(): base() {}
		public InvalidFieldTypeException(string message): base(message) {}
		public InvalidFieldTypeException(string message, Exception inner): base(message, inner) {}
#if !CF
		protected InvalidFieldTypeException(SerializationInfo info, StreamingContext context): base(info, context) {}
#endif
	}

	public sealed class MappingUtils
	{
		private MappingUtils() {}
		public static object ConvertStringToInt64(string value)
		{
			if (value == null || value.Length == 0)
				return null;
			else
				return Int64.Parse(value, NumberStyles.Integer, System.Globalization.CultureInfo.InvariantCulture);
		}

		public static object ConvertStringToBoolean(string value)
		{
			if ((value == "0") || (value == false.ToString(System.Globalization.CultureInfo.InvariantCulture))) // do not localize
				return false;
			else
				return true;
		}

		public static object ConvertStringToInt32(string value)
		{
			if (value == null || value.Length == 0)
				return null;
			else
				return Int32.Parse(value, System.Globalization.CultureInfo.InvariantCulture);
		}

		public static object ConvertStringToInt16(string value)
		{
			if (value == null || value.Length == 0)
				return null;
			else
				return Int16.Parse(value, NumberStyles.Integer, System.Globalization.CultureInfo.InvariantCulture);
		}

		public static object ConvertStringToByte(string value)
		{
			if (value == null || value.Length == 0)
				return null;
			else
				return Byte.Parse(value, NumberStyles.Integer, System.Globalization.CultureInfo.InvariantCulture);
		}

		public static object ConvertStringToSByte(string value)
		{
			if (value == null || value.Length == 0)
				return null;
			else
				return sbyte.Parse(value, NumberStyles.Integer, System.Globalization.CultureInfo.InvariantCulture);
		}

		public static object ConvertStringToChar(string value)
		{
			if (value == null || value.Length == 0)
				return null;
			else
			{
				char c = value[0];
				return c;
			}
		}

		public static object ConvertStringToDouble(string value)
		{
			if (value == null || value.Length == 0)
				return null;
			else
				return Double.Parse(value, System.Globalization.CultureInfo.InvariantCulture);
		}

		public static object ConvertStringToSingle(string value)
		{
			if (value == null || value.Length == 0)
				return null;
			else
				return Single.Parse(value, System.Globalization.CultureInfo.InvariantCulture);
		}

		public static object ConvertStringToDecimal(string value)
		{
			if (value == null || value.Length == 0)
				return null;
			else
				return Int32.Parse(value, System.Globalization.CultureInfo.InvariantCulture);
		}

		public static object ConvertStringToDateTime(string value)
		{
			if (value == null || value.Length == 0)
				return null;
			else
				return DateTime.Parse(value, System.Globalization.CultureInfo.InvariantCulture);
		}

		public static object ConvertStringToTimespan(string value)
		{
			if (value == null || value.Length == 0)
				return null;
			else
				return TimeSpan.Parse(value);
		}

		public static bool CharArraysAreEqual(System.Object value, System.Object columnValue)
		{
			if (value == null)
				return columnValue == null;
			else if (columnValue == null)
				return false;
			char[] valueArray = value as char[];
			char[] columnValueArray = columnValue as char[];
			if (valueArray.Length != columnValueArray.Length)
				return false;
			for (int i = 0; i < valueArray.Length; i++)
				if (valueArray[i] != columnValueArray[i])
					return false;
			return true;
		}

		public static bool ByteArraysAreEqual(System.Object value, System.Object columnValue)
		{
			if (value == null)
				return columnValue == null;
			else if (columnValue == null)
				return false;
			byte[] valueArray = value as byte[];
			byte[] columnValueArray = columnValue as byte[];
			if (valueArray.Length != columnValueArray.Length)
				return false;
			for (int i = 0; i < valueArray.Length; i++)
				if (valueArray[i] != columnValueArray[i])
					return false;
			return true;
		}

		// This comparsion passed on from Bold. Intended to fix rounding error?
		public static bool DateTimeIsEqual(System.Object value, System.Object columnValue)
		{
			if (value == null)
				return columnValue == null;
			else if (columnValue == null)
				return false;
			TimeSpan diff;
			diff = ((DateTime)columnValue).Subtract((DateTime)value);
			if (diff.Ticks < 0)
				diff = diff.Negate();
			return diff < new System.TimeSpan(0, 0, 2);
		}

		// This comparision passed on from Bold. Intended to fix rounding error?
		public static bool TimeSpanIsEqual(System.Object value, System.Object columnValue)
		{
			if (value == null)
				return columnValue == null;
			else if (columnValue == null)
				return false;
			TimeSpan diff;
			diff = ((TimeSpan)columnValue).Subtract((TimeSpan)value);
			if (diff.Ticks < 0)
				diff = diff.Negate();
			return diff < new System.TimeSpan(0, 0, 2);
		}
	}

	public abstract class AbstractSingleColumnAttribute
	{
		public String ColumnName(string name)
		{
			return name;
		}
		private void ThrowTypeError(object value, Type t)
		{
				throw new InvalidFieldTypeException(
					System.String.Format(CultureInfo.CurrentCulture,PersistenceStringRes.sInvalidType(
						GetType().ToString(), t.ToString(), value.ToString(), value.GetType().ToString())));
		}
		///<exception cref="InvalidFieldTypeException">Thrown if the <paramref name="value"/> is not null and not the type of <paramref name="t"/>.</exception>
		protected void EnsureType(object value, Type t)
		{
			if ((value != System.DBNull.Value) && (value != null) && (value.GetType() != t))
				ThrowTypeError(value, t);
		}

		protected TypeConverter EnsureConvertibleType(object value, Type t)
		{
			if ((value == System.DBNull.Value) || (value == null))
				return null;
			TypeConverter converter = null;
#if !CF
			converter = TypeDescriptor.GetConverter(value);
			if (!converter.CanConvertTo(t))
				ThrowTypeError(value, t);
#endif
			return converter;
		}
		
		protected object ConvertFromConverter(object value, Type type)
		{
			TypeConverter converter = EnsureConvertibleType(value, type);
#if !CF
			if (converter != null)
				return converter.ConvertTo(value, type);
			else
#endif
				ThrowTypeError(value, type);
			return null;
		}

		public abstract System.Object ColumnToValue(System.Object columnValue);

		public virtual bool IsEqual(System.Object value, System.Object columnValue)
		{
			object colValue = ColumnToValue(columnValue);
			if (value == null)
				return colValue == null;
			if (colValue == null)
				return false;
			return value.Equals(colValue);
		}
	}

	public abstract class AbstractNumericSingleColumnAttribute: AbstractSingleColumnAttribute
	{
		public virtual String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			if ((modeledValue == null) || (modeledValue.Length == 0))
				return "0"; // do not localize
			else
				return modeledValue;
		}
	}

	public abstract class AbstractStringSingleColumnAttribute: AbstractSingleColumnAttribute
	{
		public String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			if (sqlDatabaseConfig == null)
				throw new ArgumentNullException("sqlDatabaseConfig");
			return sqlDatabaseConfig.MakeStringLiteral(modeledValue);
		}
	}

	public sealed class CharAsChar: AbstractStringSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "CHAR(1)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Char));
			parameter.DbType = DbType.StringFixedLength;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = new string((System.Char)value, 1);
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToChar(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.String));
			if (DBNull.Value.Equals(columnValue) || columnValue == null)
				return null;
			else if (((System.String)columnValue).Length == 0)
				return ' '; // this is because of a bug in bdp, trailing spaces are trimmed...
			else
			{
				char c = ((string)columnValue)[0];
				return c;
			}
		}
	}

	public sealed class CharAsNChar: AbstractStringSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public string ColumnType(int length)
		{
			return "NCHAR(1)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Char));
			parameter.DbType = DbType.StringFixedLength;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = new string((System.Char)value, 1);
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToChar(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.String));
			if (DBNull.Value.Equals(columnValue))
				return null;

			if (columnValue == null) 
				return null;
			char c = ((string)columnValue)[0];
			return c;
		}
	}

	public sealed class StringAsChar: AbstractStringSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public string ColumnType(int length)
		{
			return String.Format(CultureInfo.InvariantCulture, "CHAR({0:d})", length); // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.String));
			parameter.DbType = DbType.StringFixedLength;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(value, parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.String));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class StringAsNChar: AbstractStringSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public string ColumnType(int length)
		{
			return String.Format(CultureInfo.InvariantCulture, "NCHAR({0:d})", length); // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.String));
			parameter.DbType = DbType.StringFixedLength;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(value, parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.String));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class StringAsVarChar: AbstractStringSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public string ColumnType(int length)
		{
			if (length > 0)
				return String.Format(CultureInfo.InvariantCulture, "VARCHAR({0:d})", length); // do not localize
			else
				return "VARCHAR"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.String));
			parameter.DbType = DbType.String;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(value, parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.String));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class GuidAsVarChar32: AbstractStringSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public string ColumnType(int length)
		{
			return "VARCHAR(32)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Guid));
			parameter.DbType = DbType.String;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = ((System.Guid)value).ToString("N");
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(new Guid(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.String));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return new Guid((string)columnValue);
		}
	}

	public sealed class StringAsNVarChar: AbstractStringSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public string ColumnType(int length)
		{
			return String.Format(CultureInfo.InvariantCulture, "NVARCHAR({0:d})", length); // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.String));
			parameter.DbType = DbType.String;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(value, parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.String));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class Int32AsInteger: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "INTEGER"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Int32));
			parameter.DbType = DbType.Int32;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt32(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Int32));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class UInt32AsLongInteger: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "NUMERIC(18,0)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.UInt32));
			parameter.DbType = DbType.Int64;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt64(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return ConvertFromConverter(columnValue, typeof(System.UInt32));
		}
	}

	public sealed class UInt16AsLongInteger: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "NUMERIC(18,0)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.UInt16));
			parameter.DbType = DbType.Int32;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt32(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return ConvertFromConverter(columnValue, typeof(System.UInt16));
		}
	}

	public sealed class DecimalAsDecimal: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			// DECIMAL wants (precision, scale)
			// where precision is number of digits,
			// and scale is number of decimals.
			return String.Format(CultureInfo.InvariantCulture, "DECIMAL({0:d},2)", length); // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Decimal));
			parameter.DbType = DbType.Decimal;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToDecimal(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}

	}

	public sealed class Int64AsLongInteger: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "NUMERIC(18,0)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Int64));
			parameter.DbType = DbType.Int64;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt64(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Int64));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class UInt64AsDecimal: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "NUMERIC(18,0)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.UInt64));
			parameter.DbType = DbType.Decimal;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToDecimal(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return ConvertFromConverter(columnValue, typeof(System.UInt64));
		}
	}

	public sealed class BooleanAsInteger: AbstractSingleColumnAttribute, ISingleColumnAttributemapping, INonBooleanBooleanMapping
	{
		public String ColumnType(int length)
		{
			return "INTEGER"; // do not localize
		}

		public String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			if ((modeledValue == null) || (modeledValue.Length == 0))
				return "0"; // do not localize
			else
				return modeledValue;
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Boolean));
			parameter.DbType = DbType.Int32;
			if (value == null)
				parameter.Value = DBNull.Value;
			else if ((bool)value == true)
				parameter.Value = 1;
			else
				parameter.Value = 0;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToBoolean(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Int32));
			if (columnValue == DBNull.Value)
				return null;
			else if ((System.Int32)columnValue == 0)
				return false;
			else
				return true;
		}

		public string TrueString
		{
			get { return "1"; } // do not localize
		}
	}

	public sealed class BooleanAsYNChar: AbstractSingleColumnAttribute, ISingleColumnAttributemapping, INonBooleanBooleanMapping
	{
		public String ColumnType(int length)
		{
			return "CHAR(1)"; // do not localize
		}

		public String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			if ((modeledValue == null) || (modeledValue.Length == 0))
				return "'n'"; // do not localize
			else
				return modeledValue;
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Boolean));
			parameter.DbType = DbType.String;
			if (value == null)
				parameter.Value = DBNull.Value;
			else if ((bool)value == true)
				parameter.Value = "y"; // do not localize
			else
				parameter.Value = "n"; // do not localize
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToBoolean(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.String));
			if (columnValue == DBNull.Value || columnValue == null)
				return null;
			else
				return (string.Compare(columnValue.ToString(), "N", true, CultureInfo.InvariantCulture) != 0); // ignore case // do not localize
		}

		public string TrueString
		{
			get { return "y"; } // do not localize
		}
	}

	public sealed class Int16AsSmallint: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "SMALLINT"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Int16));
			parameter.DbType = DbType.Int16;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt16(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Int16));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class ByteAsSmallInt: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "SMALLINT"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Byte));
			parameter.DbType = DbType.Int16;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = (System.Int16)(System.Byte)value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToByte(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Int16));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (System.Byte)(System.Int16)columnValue;
		}
	}

	public sealed class SByteAsSmallInt: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "SMALLINT"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.SByte));
			parameter.DbType = DbType.Int16;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = (System.Int16)(System.SByte)value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToByte(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Int16));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (System.SByte)(System.Int16)columnValue;
		}
	}

	public sealed class DoubleAsDouble: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "DOUBLE PRECISION"; // do not localize
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="parameter"/> is null</exception>
		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter"); // do not localize
			EnsureType(value, typeof(System.Double));
			parameter.DbType = DbType.Double;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToDouble(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Double));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class SingleAsSingle: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "FLOAT"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Single));
			parameter.DbType = DbType.Double;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToSingle(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Single));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	///<summary>
	///Base class mapper between a DateTime value and its persistent representation.
	///This class must be inherited.
	///</summary>
	public abstract class GenericDateTimeMapper: AbstractSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public abstract String ColumnType(int length);

		///<exception cref="ArgumentNullException">Thrown if <paramref name="sqlDatabaseConfig"/> is null</exception>
		public String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			if (sqlDatabaseConfig == null) throw new ArgumentNullException("sqlDatabaseConfig"); // do not localize
			if (((modeledValue == null) || (modeledValue.Length == 0)) && (sqlDatabaseConfig.DateTimeFormat != null) && (sqlDatabaseConfig.DateTimeFormat.Length > 0))
				return sqlDatabaseConfig.DefaultDateTime.ToString(sqlDatabaseConfig.DateTimeFormat);
			else
				return modeledValue;
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.DateTime));
			parameter.DbType = DbType.DateTime;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToDateTime(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.DateTime));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}

		public override bool IsEqual(System.Object value, System.Object columnValue)
		{
			return MappingUtils.DateTimeIsEqual(value, ColumnToValue(columnValue));
		}
	}

	public class DateTimeAsTimeStamp: GenericDateTimeMapper
	{
		public override String ColumnType(int length)
		{
			return "TIMESTAMP"; // do not localize
		}
	}

	public class DateTimeAsDateTime: GenericDateTimeMapper
	{
		public override String ColumnType(int length)
		{
			return "DATETIME"; // do not localize
		}
	}

	public abstract class GenericTimeSpanMapper: AbstractSingleColumnAttribute, ISingleColumnAttributemapping // FIXME move to Interbase namespace
	{
		private DateTime ZeroDateTime = new DateTime(1900, 1, 1, 0, 0, 0);
		public abstract String ColumnType(int length);

		public String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			return modeledValue;
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.TimeSpan));
			parameter.DbType = DbType.DateTime;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
			{
				TimeSpan ts = (TimeSpan)value;
				parameter.Value = ZeroDateTime + ts;
			}
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToTimespan(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.DateTime));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
			{
				DateTime dt = (DateTime)columnValue;
				return dt - ZeroDateTime;
			}
		}

		public override bool IsEqual(System.Object value, System.Object columnValue)
		{
			return MappingUtils.TimeSpanIsEqual(value, ColumnToValue(columnValue));
		}
	}

	public class TimeSpanAsTimeStamp: GenericTimeSpanMapper
	{
		public override String ColumnType(int length)
		{
			return "TIMESTAMP";  // do not localize
		}
	}

	public class TimeSpanAsDateTime: GenericTimeSpanMapper
	{
		public override String ColumnType(int length)
		{
			return "DATETIME";  // do not localize
		}
	}
}
